<?php
require_once 'config.php';
requireLogin();

// Get user settings
$user_settings = getUserSettings($_SESSION['user_id']);
$ai_personas = getAIPersonas();
$content_types = getContentTypes();
$notifications = getUserNotifications($_SESSION['user_id'], true);

// Handle API key save - FIXED: Check if action exists before processing
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $action = $_POST['action'];
    
    if ($action === 'save_api_key') {
        $api_key = $_POST['api_key'] ?? '';
        
        if (saveUserSettings($_SESSION['user_id'], [
            'ai_api_key' => $api_key
        ])) {
            echo json_encode(['success' => true, 'message' => 'ACCESS CODE saved successfully!']);
        } else {
            echo json_encode(['success' => false, 'message' => 'Failed to save ACCESS CODE']);
        }
        exit();
    }

    // Handle notification read - FIXED: Check if action exists
    if ($action === 'mark_notification_read') {
        $notification_id = $_POST['notification_id'] ?? '';
        if (markNotificationAsRead($notification_id, $_SESSION['user_id'])) {
            echo json_encode(['success' => true]);
        } else {
            echo json_encode(['success' => false]);
        }
        exit();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bulk AI Writer - Dashboard</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdn.jsdelivr.net/npm/@mdi/font@7.4.47/css/materialdesignicons.min.css" rel="stylesheet">
</head>
<body>
    <div class="app-container">
        <!-- Header -->
        <header class="app-header">
            <div class="logo">
                <span>🚀</span>
                <span>Bulk AI Writer</span>
            </div>
            
            <div class="header-actions">
                <button id="themeToggle" class="theme-toggle">
                    <i class="mdi mdi-weather-sunny"></i>
                </button>
                
                <?php if (isAdmin()): ?>
                <a href="123admin.php" class="btn btn-secondary btn-sm">
                    <i class="mdi mdi-shield-account"></i> Admin Panel
                </a>
                <?php endif; ?>
                
                <div class="user-menu">
                    <button class="user-button">
                        <i class="mdi mdi-account-circle"></i>
                        <span><?php echo htmlspecialchars($_SESSION['user_email']); ?></span>
                        <i class="mdi mdi-chevron-down"></i>
                    </button>
                    
                    <div class="user-dropdown">
                        <a href="#" class="dropdown-item" onclick="showSettings()">
                            <i class="mdi mdi-cog"></i>
                            Settings
                        </a>
                        <a href="forgot-password.php" class="dropdown-item">
                            <i class="mdi mdi-lock-reset"></i>
                            Change Password
                        </a>
                        <?php if (isAdmin()): ?>
                        <a href="123admin.php" class="dropdown-item">
                            <i class="mdi mdi-shield-account"></i>
                            Admin Panel
                        </a>
                        <?php endif; ?>
                        <a href="index.php?logout=1" class="dropdown-item logout">
                            <i class="mdi mdi-logout"></i>
                            Logout
                        </a>
                    </div>
                </div>
            </div>
        </header>

        <!-- Notifications -->
        <div class="notifications-container" id="notificationsContainer">
            <?php foreach ($notifications as $notification): ?>
                <div class="notification <?php echo $notification['type']; ?>" data-id="<?php echo $notification['id']; ?>">
                    <div class="notification-icon">
                        <i class="mdi mdi-<?php 
                            switch($notification['type']) {
                                case 'success': echo 'check-circle'; break;
                                case 'error': echo 'alert-circle'; break;
                                case 'warning': echo 'alert'; break;
                                default: echo 'information';
                            }
                        ?>"></i>
                    </div>
                    <div class="notification-content">
                        <h4><?php echo htmlspecialchars($notification['title']); ?></h4>
                        <p><?php echo htmlspecialchars($notification['message']); ?></p>
                    </div>
                    <button class="notification-close" onclick="closeNotification(this)">
                        <i class="mdi mdi-close"></i>
                    </button>
                </div>
            <?php endforeach; ?>
        </div>

        <!-- Main Content -->
        <main class="main-content">
            <div class="dashboard-grid">
                <!-- Control Panel -->
                <div class="control-panel">
                    <!-- API Settings -->
                    <div class="panel-section">
                        <h3><i class="mdi mdi-key"></i> ACCESS CODE AREA</h3>
                        <div class="form-group">
                            <label for="apiKey">QUILLFY ACCESS CODE</label>
                            <div class="input-with-icon">
                                <i class="mdi mdi-key-chain"></i>
                                <input type="password" id="apiKey" placeholder="Enter your QUILLFY ACCESS CODE" 
                                       value="<?php echo isset($user_settings['ai_api_key']) ? substr($user_settings['ai_api_key'], 6) : ''; ?>">
                            </div>
                            <small>We'll automatically give you access</small>
                        </div>
                        
                        <button onclick="saveAPIKey()" class="btn btn-secondary btn-full btn-sm">
                            <i class="mdi mdi-content-save"></i> Save API Key
                        </button>
                    </div>

                    <!-- Content Settings -->
                    <div class="panel-section">
                        <h3><i class="mdi mdi-tune"></i> Content Settings</h3>
                        
                        <div class="form-group">
                            <label for="contentType">Content Type</label>
                            <div class="input-with-icon">
                                <i class="mdi mdi-format-align-left"></i>
                                <select id="contentType">
                                    <?php foreach ($content_types as $type): ?>
                                        <option value="<?php echo $type['type_key']; ?>"><?php echo $type['name']; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="tone">Tone/Voice</label>
                            <div class="input-with-icon">
                                <i class="mdi mdi-voice"></i>
                                <select id="tone">
                                    <option value="professional">Professional</option>
                                    <option value="casual">Casual</option>
                                    <option value="friendly">Friendly</option>
                                    <option value="authoritative">Authoritative</option>
                                    <option value="conversational">Conversational</option>
                                    <option value="humorous">Humorous</option>
                                    <option value="persuasive">Persuasive</option>
                                    <option value="educational">Educational</option>
                                </select>
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="wordCount">Word Count</label>
                            <div class="input-with-icon">
                                <i class="mdi mdi-counter"></i>
                                <input type="number" id="wordCount" value="1000" min="100" max="5000">
                            </div>
                        </div>

                        <div class="form-group">
                            <label for="aiPersona">AI Writer Persona</label>
                            <div class="input-with-icon">
                                <i class="mdi mdi-robot"></i>
                                <select id="aiPersona">
                                    <?php foreach ($ai_personas as $persona): ?>
                                        <option value="<?php echo $persona['persona_key']; ?>"><?php echo $persona['name']; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                    </div>

                    <!-- Advanced Settings -->
                    <div class="collapsible">
                        <div class="collapsible-header">
                            <h4><i class="mdi mdi-cog"></i> Advanced AI Settings</h4>
                            <i class="mdi mdi-chevron-down"></i>
                        </div>
                        <div class="collapsible-content">
                            <div class="form-group">
                                <label>Add-ins</label>
                                <div class="checkbox-group">
                                    <label class="checkbox-item">
                                        <input type="checkbox" name="addins" value="statistics" checked>
                                        <span>Statistics</span>
                                    </label>
                                    <label class="checkbox-item">
                                        <input type="checkbox" name="addins" value="quotes" checked>
                                        <span>Quotes</span>
                                    </label>
                                    <label class="checkbox-item">
                                        <input type="checkbox" name="addins" value="examples" checked>
                                        <span>Examples</span>
                                    </label>
                                    <label class="checkbox-item">
                                        <input type="checkbox" name="addins" value="casestudies">
                                        <span>Case Studies</span>
                                    </label>
                                </div>
                            </div>

                            <div class="form-group">
                                <label for="creativity">Creativity Level</label>
                                <input type="range" id="creativity" min="0" max="100" value="75">
                                <div class="range-labels">
                                    <span>Conservative</span>
                                    <span>Balanced</span>
                                    <span>Creative</span>
                                </div>
                            </div>

                            <div class="form-group">
                                <label for="research">Research Depth</label>
                                <input type="range" id="research" min="0" max="100" value="80">
                                <div class="range-labels">
                                    <span>Basic</span>
                                    <span>Detailed</span>
                                    <span>Comprehensive</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Action Buttons -->
                    <div class="panel-section">
                        <button onclick="addTopic()" class="btn btn-secondary btn-full">
                            <i class="mdi mdi-plus"></i> Add Topic
                        </button>
                        
                        <button onclick="generateAll()" class="btn btn-primary btn-full" style="margin-top: 0.5rem;">
                            <i class="mdi mdi-robot"></i> Generate All Content
                        </button>
                        
                        <button onclick="exportAll()" class="btn btn-success btn-full" style="margin-top: 0.5rem;">
                            <i class="mdi mdi-download"></i> Export All as .doc
                        </button>
                    </div>

                    <!-- Support Section -->
                    <div class="support-section">
                        <h3><i class="mdi mdi-heart"></i> Support the Developer</h3>
                        <p>Help us improve and maintain this amazing tool</p>
                        <div class="support-buttons">
                            <button onclick="supportDeveloper()" class="btn btn-warning">
                                <i class="mdi mdi-star"></i> Support Developer
                            </button>
                            <button onclick="getHelp()" class="btn btn-secondary">
                                <i class="mdi mdi-help-circle"></i> Get Help
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Content Area -->
                <div class="content-area">
                    <div class="content-header">
                        <h2><i class="mdi mdi-pencil"></i> Content Generator</h2>
                    </div>
                    
                    <div class="content-body">
                        <!-- Topics Input -->
                        <div class="topics-container" id="topicsContainer">
                            <div class="topic-item">
                                <div class="topic-header">
                                    <div class="topic-input">
                                        <div class="form-group">
                                            <label>Topic/Title</label>
                                            <div class="input-with-icon">
                                                <i class="mdi mdi-format-title"></i>
                                                <input type="text" class="topic-title" placeholder="Enter topic or title...">
                                            </div>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Custom Prompt (Optional)</label>
                                            <div class="input-with-icon">
                                                <i class="mdi mdi-message-text"></i>
                                                <textarea class="topic-prompt" placeholder="Leave empty to use default prompt..."></textarea>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="topic-actions">
                                        <button onclick="removeTopic(this)" class="btn btn-danger btn-sm">
                                            <i class="mdi mdi-delete"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Generation Results -->
                        <div class="generation-results" id="generationResults">
                            <div class="alert alert-info">
                                <i class="mdi mdi-information"></i>
                                Add topics and click "Generate All Content" to start creating amazing content!
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Settings Modal -->
    <div id="settingsModal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Settings</h3>
                <button class="modal-close" onclick="hideSettings()">&times;</button>
            </div>
            <div class="modal-body">
                <div class="form-group">
                    <label>ACCESS CODE</label>
                    <input type="password" id="settingsApiKey" placeholder="ACCESS CODE">
                </div>
            </div>
            <div class="modal-footer">
                <button class="btn btn-primary" onclick="saveSettings()">Save Settings</button>
                <button class="btn btn-secondary" onclick="hideSettings()">Cancel</button>
            </div>
        </div>
    </div>

    <script src="script.js"></script>
    <script src="bulkwriter.js"></script>
    <script src="ai/ai.js"></script>
</body>
</html>